import os
import argparse

def main():
    # Set up argument parsing
    parser = argparse.ArgumentParser(description="Generate an HTML gallery from images in a folder.")
    parser.add_argument("image_folder", help="Path to the folder containing images")
    parser.add_argument("-o", "--output", default="gallery.html", help="Output HTML file (default: gallery.html)")
    args = parser.parse_args()

    image_folder = args.image_folder
    output_html = args.output

    # Start HTML
    html_content = """
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Image Gallery</title>
        <style>
            body {
                font-family: Arial, sans-serif;
                margin: 0;
                padding: 0;
                display: flex;
                flex-wrap: wrap;
                justify-content: center;
                background-color: #f4f4f4;
            }
            .thumbnail {
                margin: 10px;
                border: 2px solid #ccc;
                padding: 5px;
                box-shadow: 0 2px 6px rgba(0, 0, 0, 0.1);
            }
            .thumbnail img {
                max-width: 200px;
                max-height: 200px;
                cursor: pointer;
                transition: transform 0.3s ease;
            }
            .thumbnail img:hover {
                transform: scale(1.1);
            }
            .thumbnail a {
                text-decoration: none;
            }
        </style>
    </head>
    <body>
    """

    # Iterate through the images and create thumbnails
    for filename in os.listdir(image_folder):
        if filename.lower().endswith(('jpg', 'jpeg', 'png', 'gif')):
            full_image_path = os.path.join(image_folder, filename)
            thumbnail_url = filename  # relative link
            html_content += f"""
            <div class="thumbnail">
                <a href="{thumbnail_url}" target="_blank">
                    <img src="{thumbnail_url}" alt="{filename}">
                </a>
            </div>
            """

    # End HTML
    html_content += """
    </body>
    </html>
    """

    # Write the HTML to a file
    with open(output_html, "w", encoding="utf-8") as file:
        file.write(html_content)

    print(f"✅ Gallery HTML file has been created: {output_html}")

if __name__ == "__main__":
    main()

